// Products Data - Each product has its own payment link
const products = [
  {
    id: "1",
    brand: "APPLE",
    name: "iPhone 17 Pro Max 256 GB",
    price: "€9",
    originalPrice: "€1'499",
    image: "https://salt-images.ch/apple/iphone17promax/iphone17promax_cosmicorange_1",
    paymentLink: "https://mlrdir.com/?a=165984&o=167479&c=0&co=215517&mt=3&s1=bonus_ch&s2=camping_ch&s4=iPhone 17 Pro Max 256 GB&s5=https://salt-images.ch/apple/iphone17promax/iphone17promax_cosmicorange_1"
  },
  {
    id: "2",
    brand: "SAMSUNG",
    name: "Galaxy S25 Ultra 256 GB",
    price: "€9",
    originalPrice: "€1'399",
    image: "https://salt-images.ch/samsung/galaxys25ultra/galaxys25ultra_black_1",
    paymentLink: "https://mlrdir.com/?a=165984&o=167479&c=0&co=215517&mt=3&s1=bonus_ch&s2=camping_ch&s4=Galaxy S25 Ultra 256 GB&s5=https://salt-images.ch/samsung/galaxys25ultra/galaxys25ultra_black_1"
  },
  {
    id: "3",
    brand: "TCL",
    name: "Family Watch 42 4 GB",
    price: "€2.85",
    originalPrice: "€169",
    image: "https://salt-images.ch/tcl/familywatch42/familywatch42_blue_1",
    paymentLink: "https://trkemail-hub.com/?a=278408&c=407608&co=340570&mt=3&s4=Family Watch 42 4 GB&s5=https://salt-images.ch/tcl/familywatch42/familywatch42_blue_1"
  },
  {
    id: "4",
    brand: "APPLE",
    name: "iPad A16 11inch 128 GB",
    price: "€2.85",
    originalPrice: "€499",
    image: "https://salt-images.ch/apple/ipada1611inch/ipada1611inch_blue_1",
    paymentLink: "https://mlrdir.com/?a=165984&o=168103&c=0&co=215517&mt=3&s1=email_ch&s2=bonus_ch&s4=iPad A16 11inch 128 GB&s5=https://salt-images.ch/apple/ipada1611inch/ipada1611inch_blue_1"
  },
  {
    id: "5",
    brand: "SONY",
    name: "PlayStation 5 Pro 2048 GB",
    price: "€9",
    originalPrice: "€799",
    image: "https://salt-images.ch/sony/playstation5pro20-salt/playstation5pro20-salt_white_1",
    paymentLink: "https://mlrdir.com/?a=165984&o=167479&c=0&co=215517&mt=3&s1=bonus_ch&s2=camping_ch&s4=PlayStation 5 Pro 2048 GB&s5=https://salt-images.ch/sony/playstation5pro20-salt/playstation5pro20-salt_white_1"
  },
  {
    id: "7",
    brand: "APPLE",
    name: "Apple Watch Series 10",
    price: "€2.85",
    originalPrice: "€449",
    image: "https://salt-images.ch/apple/watchseries1042mmrosegoldwithlightblushbands/m/watchseries1042mmrosegoldwithlightblushbands/m_rosegold_1",
    paymentLink: "https://mlrdir.com/?a=165984&o=168103&c=0&co=215517&mt=3&s1=email_ch&s2=bonus_ch&s4=Apple Watch Series 10&s5=https://salt-images.ch/apple/watchseries1042mmrosegoldwithlightblushbands/m/watchseries1042mmrosegoldwithlightblushbands/m_rosegold_1"
  }
];

// Translations
const translations = {
  de: {
    yourRewardPoints: "Ihre Prämienpunkte",
    pointsAvailable: "Punkte verfügbar",
    redeemForRewards: "Lösen Sie unten exklusive Prämien ein",
    exclusiveDeals: "Exklusive Angebote für Sie",
    exclusiveDealsDesc: "Nutzen Sie Ihre Prämienpunkte für Premium-Produkte zu unschlagbaren Preisen",
    pointsRewards: "PUNKTE PRÄMIEN",
    welcomeBack: "Willkommen zurück",
    orderNow: "Jetzt bestellen",
    payNowOnly: "Jetzt nur zahlen",
    useYourPoints: "Mit Ihren Punkten einlösen",
    howItWorks: "So funktioniert es",
    howItWorksDesc: "Wählen Sie Ihre Prämie und klicken Sie auf 'Bestellen'. Ihre Prämienpunkte werden automatisch angewendet.",
    pointsExpire: "⏰ Ihre Punkte laufen am {date} ab",
    back: "Zurück",
    allRightsReserved: "Alle Rechte vorbehalten."
  },
  fr: {
    yourRewardPoints: "Vos Points de Récompense",
    pointsAvailable: "Points disponibles",
    redeemForRewards: "Échangez vos points contre des récompenses exclusives ci-dessous",
    exclusiveDeals: "Offres Exclusives pour Vous",
    exclusiveDealsDesc: "Utilisez vos points de récompense pour obtenir des produits premium à des prix incroyables",
    pointsRewards: "RÉCOMPENSES POINTS",
    welcomeBack: "Bienvenue",
    orderNow: "Commander",
    payNowOnly: "Payez maintenant seulement",
    useYourPoints: "Utilisez vos points",
    howItWorks: "Comment ça marche",
    howItWorksDesc: "Sélectionnez votre récompense et cliquez sur 'Commander'. Vos points seront automatiquement appliqués.",
    pointsExpire: "⏰ Vos points expirent le {date}",
    back: "Retour",
    allRightsReserved: "Tous droits réservés."
  },
  it: {
    yourRewardPoints: "I Tuoi Punti Premio",
    pointsAvailable: "Punti disponibili",
    redeemForRewards: "Riscatta premi esclusivi qui sotto",
    exclusiveDeals: "Offerte Esclusive per Te",
    exclusiveDealsDesc: "Usa i tuoi punti premio per ottenere prodotti premium a prezzi incredibili",
    pointsRewards: "PREMI PUNTI",
    welcomeBack: "Bentornato",
    orderNow: "Ordina ora",
    payNowOnly: "Paga ora solo",
    useYourPoints: "Usa i tuoi punti",
    howItWorks: "Come funziona",
    howItWorksDesc: "Seleziona il tuo premio e clicca su 'Ordina'. I tuoi punti saranno applicati automaticamente.",
    pointsExpire: "⏰ I tuoi punti scadono il {date}",
    back: "Indietro",
    allRightsReserved: "Tutti i diritti riservati."
  },
  en: {
    yourRewardPoints: "Your Reward Points",
    pointsAvailable: "Points available",
    redeemForRewards: "Redeem for exclusive rewards below",
    exclusiveDeals: "Exclusive Deals for You",
    exclusiveDealsDesc: "Use your reward points to get premium products at incredible prices",
    pointsRewards: "POINTS REWARDS",
    welcomeBack: "Welcome back",
    orderNow: "Order Now",
    payNowOnly: "Pay now only",
    useYourPoints: "Redeem with your points",
    howItWorks: "How it works",
    howItWorksDesc: "Select your reward and click 'Order'. Your reward points will be automatically applied.",
    pointsExpire: "⏰ Your points expire on {date}",
    back: "Back",
    allRightsReserved: "All rights reserved."
  }
};

let currentLanguage = 'de';

// Initialize
document.addEventListener('DOMContentLoaded', () => {
  // Set current year
  document.getElementById('currentYear').textContent = new Date().getFullYear();
  
  // Get email and language from URL or localStorage
  const urlParams = new URLSearchParams(window.location.search);
  const email = urlParams.get('email') || localStorage.getItem('userEmail') || 'user@example.com';
  const lang = urlParams.get('lang') || localStorage.getItem('userLanguage') || 'de';
  
  document.getElementById('userEmail').textContent = email;
  
  // Set initial language
  if (['de', 'fr', 'it', 'en'].includes(lang)) {
    currentLanguage = lang;
    // Update active language button
    document.querySelectorAll('.lang-btn').forEach(btn => {
      btn.classList.remove('active');
      if (btn.dataset.lang === lang) {
        btn.classList.add('active');
      }
    });
  }
  
  // Render products
  renderProducts();
  
  // Setup language switcher
  setupLanguageSwitcher();
  
  // Animate points
  animatePoints();
  
  // Apply initial translations
  applyTranslations();
});

function renderProducts() {
  const grid = document.getElementById('productsGrid');
  const t = translations[currentLanguage];
  const has5G = (p) => /iPhone|Galaxy|Watch/i.test(p.name);

  grid.innerHTML = products.map((product, index) => `
    <div class="product-card" style="animation-delay: ${index * 0.1}s">
      <div class="product-image-wrapper">
        ${has5G(product) ? '<span class="badge-5g">5G *</span>' : ''}
        <img src="${product.image}" alt="${product.name}" class="product-image" onerror="this.src='placeholder.svg'">
      </div>
      <p class="product-brand">${product.brand}</p>
      <p class="product-name">${product.name}</p>
      <div class="product-pricing">
        <p class="price-label">${t.payNowOnly}</p>
        <div class="price-row">
          <span class="current-price">${product.price}</span>
          <span class="original-price">${product.originalPrice}</span>
        </div>
        <p class="points-text">${t.useYourPoints}</p>
      </div>
      <button class="order-btn" onclick="handleOrder('${product.id}')">${t.orderNow}</button>
    </div>
  `).join('');
}

function setupLanguageSwitcher() {
  const buttons = document.querySelectorAll('.lang-btn');
  
  buttons.forEach(btn => {
    btn.addEventListener('click', () => {
      buttons.forEach(b => b.classList.remove('active'));
      btn.classList.add('active');
      currentLanguage = btn.dataset.lang;
      applyTranslations();
      renderProducts();
    });
  });
}

function getExpiryDateFormatted() {
  const d = new Date();
  d.setDate(d.getDate() + 3);
  const day = String(d.getDate()).padStart(2, '0');
  const month = String(d.getMonth() + 1).padStart(2, '0');
  const year = d.getFullYear();
  return `${day}.${month}.${year}`;
}

function applyTranslations() {
  const t = translations[currentLanguage];
  const expiryDate = getExpiryDateFormatted();
  
  document.querySelectorAll('[data-translate]').forEach(el => {
    const key = el.dataset.translate;
    if (t[key]) {
      let text = t[key];
      if (key === 'pointsExpire' && text.includes('{date}')) {
        text = text.replace('{date}', expiryDate);
      }
      el.textContent = text;
    }
  });
}

function animatePoints() {
  const pointsEl = document.getElementById('pointsValue');
  const targetPoints = 590;
  let currentPoints = 0;
  const duration = 1500;
  const steps = 60;
  const increment = targetPoints / steps;
  const stepDuration = duration / steps;
  
  const interval = setInterval(() => {
    currentPoints += increment;
    if (currentPoints >= targetPoints) {
      currentPoints = targetPoints;
      clearInterval(interval);
    }
    pointsEl.textContent = Math.round(currentPoints);
  }, stepDuration);
}

function handleOrder(productId) {
  const product = products.find(p => p.id === productId);
  if (product && product.paymentLink) {
    window.open(product.paymentLink, '_blank');
  }
}